/*:
 * @plugindesc Plays a stepping noise when the player moves, based on the tile type and walking speed.
 * @target MZ
 * @param grassSound
 * @text Grass Sound
 * @desc Sound effect for walking on grass.
 * @default GrassStep
 * @type file
 * @dir audio/se
 * @param woodSound
 * @text Wood Sound
 * @desc Sound effect for walking on wood.
 * @default WoodStep
 * @type file
 * @dir audio/se
 * @param stoneSound
 * @text Stone Sound
 * @desc Sound effect for walking on stone.
 * @default StoneStep
 * @type file
 * @dir audio/se
 * @param soundVolume
 * @text Sound Volume
 * @desc Volume of the stepping sounds.
 * @default 90
 * @type number
 * @min 0
 * @max 100
 * @param stepDelay
 * @text Step Delay
 * @desc Delay in frames between step sounds.
 * @default 24
 * @type number
 * @min 1
 * @max 120
 */

(function() {
    const parameters = PluginManager.parameters('TileStepSounds');
    const grassSound = String(parameters['grassSound'] || 'GrassStep');
    const woodSound = String(parameters['woodSound'] || 'WoodStep');
    const stoneSound = String(parameters['stoneSound'] || 'StoneStep');
    const soundVolume = Number(parameters['soundVolume'] || 90);
    const defaultStepDelay = 24;
    const sprintStepDelay = 18;
    const sprintingSwitchId = 35; // ID for "Player Is Sprinting" switch

    const tileSoundMap = {
        1: grassSound,   // Grass tile
        2: woodSound,    // Wood tile
        3: stoneSound    // Stone tile
    };

    let lastStepFrame = 0;

    const alias_Game_Player_update = Game_Player.prototype.update;
    Game_Player.prototype.update = function(sceneActive) {
        alias_Game_Player_update.call(this, sceneActive);
        this.updateStepSound();
    };

    Game_Player.prototype.updateStepSound = function() {
        const currentFrame = Graphics.frameCount;
        const stepDelay = $gameSwitches.value(sprintingSwitchId) ? sprintStepDelay : defaultStepDelay;
        if (this.isMoving() && currentFrame >= lastStepFrame + stepDelay) {
            this.playStepSound();
            lastStepFrame = currentFrame;
        }
    };

    const alias_Game_Player_moveByInput = Game_Player.prototype.moveByInput;
    Game_Player.prototype.moveByInput = function() {
        const sprinting = $gameSwitches.value(sprintingSwitchId);
        const speed = sprinting ? 5 : 4;
        if (this.moveSpeed() !== speed) {
            this.setMoveSpeed(speed);
        }
        alias_Game_Player_moveByInput.call(this);
    };

    Game_Player.prototype.playStepSound = function() {
        const regionId = $gameMap.regionId(this.x, this.y);
        const stepSound = tileSoundMap[regionId];
        if (stepSound) {
            const se = {
                name: stepSound,
                volume: soundVolume,
                pitch: 90 + Math.floor(Math.random() * 21), // Random pitch between 90 and 110
                pan: 0
            };
            AudioManager.playSe(se);
        }
    };
})();
